package org.example.mobiledeviceslistingjob;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.device.api.IDevice;
import net.bluemind.domain.api.IDomains;
import net.bluemind.scheduledjob.api.JobExitStatus;
import net.bluemind.scheduledjob.api.JobKind;
import net.bluemind.scheduledjob.scheduler.IScheduledJob;
import net.bluemind.scheduledjob.scheduler.IScheduledJobRunId;
import net.bluemind.scheduledjob.scheduler.IScheduler;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class MobileDevicesListingJob implements IScheduledJob{

	private Logger logger = LoggerFactory.getLogger(MobileDevicesListingJob.class);
	
	@Override
	public void tick(IScheduler sched, boolean plannedOrForced, String domainName, Date startDate) throws ServerFault {		
		if (!plannedOrForced){
			// some jobs tasks depend on some trigger and not on scheduling.
			// the tick method is executed for the check to occur but this is not our case
			logger.debug("Not planned nor forced: not executing.");
			return;
		}
		IScheduledJobRunId slot = sched.requestSlot("global.virt", this, startDate);
		if (slot == null){
			logger.error("Could not get an execution slot!");
			return;
		}
		
		// logger will write in bm core logs (/var/log/bm/core.log)
		logger.info("Executing MobileDevicesListingJob");
		// sched will write in the execution report, that you can send by mail in the Job setup UI
		sched.info(slot, "en", "Collecting mobile devices data for all users...\n");
		// write header row for the data to come
		sched.info(slot, "en", "device; isWipe; lastSync; user; domain");
		// initialize client for domain service
		IDomains domainService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IDomains.class);
		// loop on all domains
		domainService.all().stream().forEach(domain -> {
			// initialize client for user service
			IUser userService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IUser.class, domain.uid);
			// loop on all users
			userService.allUids().stream().forEach(userUID -> {
				// grab full details for user
				ItemValue<User> user = userService.getComplete(userUID);				
				// initialize device service for each user
				try {
					IDevice deviceService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
							.instance(IDevice.class, userUID);
					// loop on all devices
					deviceService.list().values.stream().forEach(device -> {
						// collect info for this device
						List<String> deviceInfo = new ArrayList<>();
						deviceInfo.add(device.displayName);
						deviceInfo.add(Boolean.toString(device.value.isWipe));
						deviceInfo.add(device.value.lastSync.toString());
						deviceInfo.add(user.displayName);
						deviceInfo.add(domain.displayName);
						// write a line in the report					
						sched.info(slot, "en", String.join("; ", deviceInfo));
					});;
				} catch (ServerFault exception){
					// Skipping this user since she doesn't have a "device" container
				}
			});

		});
		sched.finish(slot, JobExitStatus.SUCCESS);
		
	}

	@Override
	public JobKind getType() { 
		return JobKind.GLOBAL;
	}

	@Override
	public String getDescription(String locale) {
		return "Lists all devices belonging to all users of all domains";
	}

	@Override
	public String getJobId() {
		return "org.example.mobiledeviceslistingjob.MobileDevicesListingJob";
	}

	@Override
	public Set<String> getLockedResources() {
		return Collections.emptySet();
	}	
	

}
